#define DRV_NAME	"8211"
#define DRV_VERSION	"1.05"
#define DRV_RELDATE	"June 24, 2003"

#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/random.h>
#include <linux/wireless.h>
#include <linux/if_arp.h>
#include <linux/delay.h>
#include <asm/delay.h>
#include <asm/processor.h>	/* Processor type for cache alignment. */
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/unaligned.h>
#include <asm/uaccess.h>
#include "STA_Module.h"
#include "lm.h"
#include "Wlan_Mac.h"
#include "adm8201.h"
#include <linux/mii.h>
#include <linux/module.h>
#include <linux/init.h>
static char version[] __devinitdata =
	"ADM8211 Linux driver version " DRV_VERSION " (" DRV_RELDATE ")\n";
	
/* Maximum events (Rx packets, etc.) to handle at each interrupt. */
static int max_interrupt_work = 25;

int csr0;

// UM function
extern void UM_Init_Module(STA_MDL *);
extern void UM_MmIndicate(STA_MDL *, int, UINT8 *, int);
extern void UM_Free_Memory(STA_MDL *);
extern UINT8* UM_3to11(STA_MDL *, UINT8*, int*, UINT8*, int*);
extern UINT8* UM_11to3(STA_MDL *, UINT8*, int*);
extern void UM_Clear_SRAM(STA_MDL *);
extern void UM_Software_WEP(STA_MDL *, UINT8*, int*);
extern void MM_TimeSynchronize(STA_MDL *);
extern void UM_Query_Information(STA_MDL *, UINT32, UINT8 *, int *);
extern void UM_Set_Information(STA_MDL *, UINT32, UINT8 *, int *);
extern void MM_Set_IBSS(STA_MDL * pSTA_Module, int );
extern void MM_CreateAdhocArea(STA_MDL * pSTA_Module);

// RF function
extern void RF_SetBBPReadWriteAddr(LM_DEVICE_INFO * pLM_Device);
extern void RF_InitializeBBP(LM_DEVICE_INFO * pLM_Device);
extern void RF_SetChannel(LM_DEVICE_INFO * pLM_Device, BYTE channel);
extern void RF_SetAntenna(LM_DEVICE_INFO * pLM_Device , int Antenna );

// LM function
extern void ChipReset(LM_DEVICE_INFO * pLM_Device);
extern void Software_Init(LM_DEVICE_INFO * pLM_Device);
extern void Hardware_Init(LM_DEVICE_INFO * pLM_Device);
extern void DefragmentPacketAssemble(LM_DEVICE_INFO * pLM_Device , BYTE *data, short pkt_len);


MODULE_AUTHOR("ADMtek");
MODULE_DESCRIPTION("ADM8211 Wireless netword card driver");
MODULE_LICENSE("Proprietary");

static int __devinit adm8211_init_one (struct pci_dev *pdev,
				     const struct pci_device_id *ent);			  
static struct pci_device_id pci_id_tbl[] __devinitdata = {
	{ 0x1317, 0x8211, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ 0x1317, 0x8201, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ } /* terminate list */
};
MODULE_DEVICE_TABLE(pci, pci_id_tbl);

static void CheckForHangTimer(unsigned long data);

enum tbl_flag {
    HAS_MII = 1,
    MC_HASH_ONLY = 0x20,
    ADM8201_MAC_ADDR = 0x0800,
};

enum adm8211_busconfig_bits {
	MWI			= (1 << 24),
	MRL			= (1 << 23),
	MRM			= (1 << 21),
	CALShift		= 14,
	BurstLenShift		= 8,
};

struct adm8201_chip_table adm8201_tbl[] = {
    {"ADMtek Wireless", 256, 0x040180c5, HAS_MII | MC_HASH_ONLY | ADM8201_MAC_ADDR, CheckForHangTimer},
	{0},};

static int read_eeprom(long ioaddr, int location, int addr_len);
static int adm8201_start_xmit(struct sk_buff *skb, struct net_device *dev);
static void adm8201_interrupt(int irq, void *dev_instance, struct pt_regs *regs);
static void adm8201_tx_interrupt(struct net_device *dev);
static int adm8201_rx_interrupt(struct net_device *dev);

static struct net_device_stats *adm8201_get_stats(struct net_device *dev);

static void adm8201_init_ring(struct net_device *dev);
static void empty_rings(struct net_device *dev);

/* jackl add funcitons */
static void adm8201_Init(struct net_device *dev);
static void DisableInterrupt(struct net_device *dev);
static void EnableInterrupt(struct net_device *dev);


static int adm8201_ioctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int adm8201_open(struct net_device *dev);
static int adm8201_close(struct net_device *dev);

struct iw_statistics *adm8201_get_wireless_stats (struct net_device *dev);

/*this is LM prototype*/
int LOCAL_Timer_func(STA_MDL * pSTA_Module);
void LM_Set_Timer(STA_MDL * pSTA_Module);
void LM_Del_Timer(void);
void LM_Delay(WORD x);
UINT8 *LM_Get_Memory(int Mem_len);
void LM_Send_Packet(STA_MDL * pSTA_Module);
void LM_Write_CSR(STA_MDL * pSTA_Module, BYTE CSR_Offset, unsigned long Value32);
UINT32 LM_Read_CSR(STA_MDL * pSTA_Module, BYTE CSR_Offset);
void LM_Spin_Lock(void);
void LM_Spin_Unlock(void);
UINT8 LM_Random(void);
void LM_Set_Channel(STA_MDL * pSTA_Module , BYTE channel);
void LM_SetAntenna(STA_MDL * pSTA_Module , int Antenna );

void LM_Adm8201_Reset(struct net_device *dev);
//Jeffrey06232003
void LM_CheckDxferState(struct net_device *dev);   

/* Global variable for LM*/
struct timer_list LM_timer_list;
spinlock_t lock;

/* The frequency of each channel in MHz */
const long channel_frequency[] = {
241200000, 241700000, 242200000, 242700000, 243200000, 243700000, 244200000,
244700000, 245200000, 245700000, 246200000, 246700000, 247200000, 248400000
};


static int __devinit adm8211_init_one (struct pci_dev *pdev,
				     const struct pci_device_id *ent)
{
    struct net_device *dev;
    struct adm8201_private *pDevice;
    /* See note below on the multiport cards. */
    static unsigned char last_phys_addr[6] =
	{ 'A', 'D', 'M', 't', 'e', 'k' };
    static int last_irq = 0;
    u8 chip_rev;
    int i;
    unsigned short sum;
    u8 ee_data[EEPROM_SIZE];
    /*Jeffrey : for EEPROM declaration */
    int sa_offset;
    int ee_addr_size;
    int eeprom_word_cnt;
    /*NEW PCI declaration*/
    long ioaddr;
    int irq;
    int chip_idx = ent->driver_data;

#ifdef ioctl_debug
    printk("adm8201 probe \n");
#endif
    printk (KERN_INFO "%s", version);

    csr0 = PCI_BURST_LENGTH_8DW;

    i = pci_enable_device(pdev);

    if (i) 
    {
	printk( "Cannot enable ADM8211 board , aborting\n");
	return i;
    }

    ioaddr = pci_resource_start (pdev, 0);
    irq = pdev->irq;
	
    /* alloc_etherdev ensures aligned and zeroed private structures */
    dev = alloc_etherdev (sizeof (*pDevice));
	
    if (!dev) 
    {
	printk ("ethernet device alloc failed, aborting\n");
	return -ENOMEM;
    }

   
    if (pci_resource_len (pdev, 0) < adm8201_tbl[chip_idx].io_size) {
   	printk ("%s: I/O region (0x%lx@0x%lx) too small, "
			"aborting\n", pdev->slot_name,
	pci_resource_len (pdev, 0),
	pci_resource_start (pdev, 0));
	goto ERR_OUT;
    }

    if (pci_request_regions (pdev, "8211"))
	goto ERR_OUT;
		
    pci_read_config_byte (pdev, PCI_REVISION_ID, &chip_rev);	
   
    SET_MODULE_OWNER(dev);	

    pDevice = dev->priv;

    pci_set_master(pdev);

/*Jeffrey03042003: Power on issue*/
    outl(0, ioaddr + 0x04);  
    outl(0x00000001, ioaddr + 0x04); 
    mdelay(100);

/* Clear the missed-packet counter. */
    inl(ioaddr + CSR8);
    sum=0;

    put_unaligned(le32_to_cpu(inl(ioaddr + 0x94)), (u32 *) dev->dev_addr);
    put_unaligned(le16_to_cpu(inl(ioaddr + 0x98)), (u16 *) (dev->dev_addr + 4));
    for (i = 0; i < 6; i++)
	sum += dev->dev_addr[i];

    sa_offset = 0;
    ee_addr_size = read_eeprom(ioaddr, 0xff, 8) & 0x40000 ? 8 : 6;
    eeprom_word_cnt = 1 << ee_addr_size;

    for (i = 0; i < eeprom_word_cnt; i++)
    {
	((u16 *) ee_data)[i] = le16_to_cpu(read_eeprom(ioaddr, i,ee_addr_size));
	memcpy(&pDevice->Eeprom, ee_data, sizeof(pDevice->Eeprom));
    }
//Jeffrey01062003 
    for (i = 0; i < eeprom_word_cnt; i++)
    {
	((u16 *) ee_data)[i] = le16_to_cpu(read_eeprom(ioaddr, i,ee_addr_size));
	memcpy(&pDevice->LM_Device.Eeprom, ee_data, sizeof(pDevice->Eeprom));
    }
//Read transceiver type
    switch (pDevice->Eeprom.SpecificRFType)
    {
	case 1 :
		pDevice->LM_Device.TransceiverType = RFMD2958;
		break;
	//Jeffrey06232003
	case 2 :
		pDevice->LM_Device.TransceiverType = RFMD2958_RF3000_CONTROL_POWER;
		break;
	
    }

    for (i=0;i<7;i++)
    {
	pDevice->LM_Device.TxPwr[i] = pDevice->Eeprom.TxPwr[i];
    }

    if (sum == 0 || sum == 6 * 0xff)
    {
	for (i = 0; i < 5; i++)
	    	dev->dev_addr[i] = last_phys_addr[i];
	dev->dev_addr[i] = last_phys_addr[i] + 1;
     }
    printk(KERN_INFO "8211: Hardware Address");	
    for (i = 0; i < 6; i++)
    {
	 printk("%c%2.2X", i ? ':' : ' ', last_phys_addr[i] = dev->dev_addr[i]);
    }
    printk(", IRQ %d.\n", irq);
    last_irq = irq;


    dev->base_addr = ioaddr;
    dev->irq = irq;

    pDevice->pci_dev = pdev;
    pDevice->chip_id = chip_idx;
    pDevice->revision = chip_rev;
    /*revision*/
    pDevice->LM_Device.RevisionStep = chip_rev;

    pDevice->flags = adm8201_tbl[chip_idx].flags;
    pDevice->csr0 = csr0;

    pDevice->LM_Device.CSR0_Values = csr0;


    /* The adm8201-specific entries in the device structure. */

    dev->open = &adm8201_open;
    dev->hard_start_xmit = &adm8201_start_xmit;
    dev->stop = &adm8201_close;
    dev->get_stats = &adm8201_get_stats;
    dev->do_ioctl = &adm8201_ioctl;

    /*wireless extension*/
    dev->get_wireless_stats = &adm8201_get_wireless_stats;
    
    if (register_netdev(dev))
 	 goto ERR_OUT;

    printk(KERN_INFO "%s: %s rev %d at %#3lx\n",
	       dev->name, adm8201_tbl[chip_idx].chip_name, chip_rev, ioaddr);
	       
    pci_set_drvdata(pdev, dev);
    
    spin_lock_init(&lock); 

    pDevice->STA_Module.MDL_pDrvObj = (unsigned long*)dev;
    pDevice->LM_Device.MDL_pDrvObj = (unsigned long*)(&(pDevice->STA_Module));
    Software_Init(&(pDevice->LM_Device));

    return 0;
    
ERR_OUT:

    kfree (dev);
        return -ENODEV;
	    
}



#define EE_SHIFT_CLK	0x02	
#define EE_CS		0x01	
#define EE_DATA_WRITE	0x04	
#define EE_WRITE_0	0x01
#define EE_WRITE_1	0x05
#define EE_DATA_READ	0x08	
#define EE_ENB		(0x4800 | EE_CS)

#define eeprom_delay()	inl(ee_addr)

#define EE_READ_CMD		(6)

static int read_eeprom(long ioaddr, int location, int addr_len)
{
    int i;
    unsigned retval = 0;
    long ee_addr = ioaddr + CSR9;
    int read_cmd = location | (EE_READ_CMD << addr_len);

    outl(EE_ENB & ~EE_CS, ee_addr);
    outl(EE_ENB, ee_addr);

    /* Shift the read command bits out. */
    for (i = 4 + addr_len; i >= 0; i--) {
	short dataval = (read_cmd & (1 << i)) ? EE_DATA_WRITE : 0;
	outl(EE_ENB | dataval, ee_addr);
	eeprom_delay();
	outl(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
	eeprom_delay();
	retval = (retval << 1) | ((inl(ee_addr) & EE_DATA_READ) ? 1 : 0);
    }
    outl(EE_ENB, ee_addr);
    eeprom_delay();

    for (i = 16; i > 0; i--) {
	outl(EE_ENB | EE_SHIFT_CLK, ee_addr);
	eeprom_delay();
	retval = (retval << 1) | ((inl(ee_addr) & EE_DATA_READ) ? 1 : 0);
	outl(EE_ENB, ee_addr);
	eeprom_delay();
    }

    /* Terminate the EEPROM access. */
    outl(EE_ENB & ~EE_CS, ee_addr);
    return retval;
}

struct iw_statistics *adm8201_get_wireless_stats(struct net_device *dev)
{
	
	struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
	pDevice->wstats.qual.qual = pDevice->linkquality;
	pDevice->wstats.qual.noise = 0;
        pDevice->wstats.qual.updated = 10;

	return (&pDevice->wstats);
}

static int adm8201_open(struct net_device *dev)
{

	struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
	long ioaddr = dev->base_addr;
#ifdef ioctl_debug
	printk("adm8211 open\n");
#endif

	pDevice->LM_Device.Start_ADM8211=1;

	//if not request irq then unload driver
	if (request_irq(dev->irq, &adm8201_interrupt, SA_SHIRQ, dev->name, dev))
	{
		return -EAGAIN;
	}
	adm8201_Init(dev);

 /*Jeffrey : 11012002*/
	init_timer(&LM_timer_list);
	
	outl(0x00002002, ioaddr + CSR6);  // start tx and rx
	EnableInterrupt(dev);
	pDevice->LM_Device.UmInitModule_Flag=0;
	UM_Init_Module(&pDevice->STA_Module);

    return 0;
}

static void CheckForHangTimer(unsigned long data)
{
    struct net_device *dev = (struct net_device *) data;
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    int next_tick = 2 * HZ;
    int i,j;
/*Jeffrey: for link quality*/
    
	if ( (pDevice->STA_Module.MDL_Current_Action == ALL_COMPLETE ) &&
	 (pDevice->STA_Module.MDL_Operation_Mode == 1 ) )
	{

		pDevice->linkquality=(int)((pDevice->beaconcount*(pDevice->STA_Module.MDL_AP_Info[pDevice->STA_Module.MDL_AP_Inuse_Index].Beacon_Interval))/20);
		if(pDevice->linkquality > 96)
		{
			pDevice->linkquality = 100;
		}	
    		pDevice->beaconcount = 0;//reset beacon count
	}
 /*end of link quality*/

	if ( (pDevice->STA_Module.MDL_Current_Action == ALL_COMPLETE ) &&
	 (pDevice->STA_Module.MDL_Operation_Mode == Ndis802_11IBSS ) )
	{

		if (pDevice->LM_Device.BeaconCount != 0)
		{
			 pDevice->LM_Device.BeaconCount = 0;
		}
		else
		{
			if (pDevice->LM_Device.LostBeaconFlag == 0)
				pDevice->LM_Device.LostBeaconFlag = 1;
			else
			{
				netif_stop_queue(dev);
				LM_Adm8201_Reset(dev);
				pDevice->LM_Device.LostBeaconFlag = 0;
				netif_start_queue(dev);
			}
		}
	}

	if( (pDevice->STA_Module.MDL_Assoc_Complete == 1) ||
	    (pDevice->STA_Module.MDL_Current_Action == ALL_COMPLETE) )
	{

		if (pDevice->LM_Device.Start_Tx == 0)
		{
			#ifdef adm8201_debug
				printk(" start to send packet \n");
			#endif
			netif_start_queue(dev);
   			pDevice->LM_Device.Start_Tx=1;
		}

		for (i = 0; i < 3; i++)
		{
			if ((pDevice->RxDef[i].Inside == 1) && (pDevice->RxDef[i].TimeStatus == 1))
			{
				#ifdef adm8201_debug
					printk("Clean from queue \n");
				#endif
				pDevice->LM_Device.RxDef[i].TimeStatus = 0;
				pDevice->LM_Device.RxDef[i].fra_SeqNum = 0;
				pDevice->LM_Device.RxDef[i].fra_FraNum = 0;
				pDevice->LM_Device.RxDef[i].Inside = 0;
				for (j = 0; j < 6; j++)
				{
		    		pDevice->LM_Device.RxDef[i].MAC[j] = 0x00;
				}
				if ( pDevice->LM_Device.NumofQueue != 0)
					pDevice->LM_Device.NumofQueue--;
				pDevice->LM_Device.RxDef[i].data_len=0;
				pDevice->LM_Device.CompleteQueueNum=0;
	    	}

			if ((pDevice->LM_Device.RxDef[i].Inside == 1) && (pDevice->LM_Device.RxDef[i].TimeStatus == 0))
			{
				pDevice->LM_Device.RxDef[i].TimeStatus = 1;
	   		}
		}
    }
//Jeffrey06232003
    if ((pDevice->STA_Module.MDL_Current_Action == ALL_COMPLETE) &&
        (pDevice->LM_Device.RevisionStep == REVISION_BA) &&
        (pDevice->RxPacketCountPer2S == 0) )
    {
	LM_CheckDxferState(dev);		
    }	
    pDevice->RxPacketCountPer2S = 0;	

    pDevice->timer.expires = jiffies + next_tick;
    add_timer(&pDevice->timer);
}

/* Initialize the Rx and Tx rings */
static void adm8201_init_ring(struct net_device *dev)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    long ioaddr = dev->base_addr;
    int i;

    pDevice->rx_dead = pDevice->tx_full = 0;
    pDevice->cur_rx = pDevice->cur_tx = 0;
    pDevice->dirty_rx = pDevice->dirty_tx = 0;

    for (i = 0; i < RX_RING_SIZE; i++)
    {
	pDevice->rx_ring[i].status = 0x00000000;
	pDevice->rx_ring[i].length = cpu_to_le32(PKT_BUF_SZ);
	pDevice->rx_ring[i].buffer2 = virt_to_le32desc(&pDevice->rx_ring[i+1]);
	pDevice->rx_skbuff[i] = NULL;
    }

    pDevice->rx_ring[i - 1].length = cpu_to_le32(PKT_BUF_SZ | DESC_RING_WRAP);
    pDevice->rx_ring[i - 1].buffer2 = virt_to_le32desc(&pDevice->rx_ring[0]);

    for (i = 0; i < RX_RING_SIZE; i++)
    {
	pDevice->RDES1[i] = le32_to_cpu(pDevice->rx_ring[i].length);
    }
	
	// allocate Rx buffer
    for (i = 0; i < RX_RING_SIZE; i++)
    {
	struct sk_buff *skb = dev_alloc_skb(PKT_BUF_SZ);
	pDevice->rx_skbuff[i] = skb;
	if (skb == NULL)
		break;
	skb->dev = dev;		/* Mark as being used by this device. */
	pDevice->rx_ring[i].status = cpu_to_le32(DescOwned);
	pDevice->rx_ring[i].buffer1 = virt_to_le32desc(skb->tail);
    }
    pDevice->dirty_rx = (unsigned int) (i - RX_RING_SIZE);

    for (i = 0; i < TX_RING_SIZE; i++)
    {
	pDevice->tx_skbuff[i] = 0;
	pDevice->tx_ring[i].status = 0x00000000;
    }

    outl(virt_to_bus(pDevice->rx_ring), ioaddr + CSR3);
    outl(virt_to_bus(pDevice->tx_ring), ioaddr + CSR4);

}

static int adm8201_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    int entry, q_used_cnt;
    u32 flag;
    int Length1,Length2;

    /* Calculate the next Tx descriptor entry. */
    entry = pDevice->cur_tx % TX_RING_SIZE;

    q_used_cnt = pDevice->cur_tx - pDevice->dirty_tx;

    pDevice->tx_skbuff[entry] = skb;

    Length2=skb->len;


	//convert 802.3 header to 802.11
    skb->data = (unsigned char *)UM_3to11(&(pDevice->STA_Module), (unsigned char *)&(pDevice->MSDU[entry].Content), &Length1, skb->data, &Length2);

    pDevice->tx_ring[entry].buffer1 = virt_to_le32desc(&(pDevice->MSDU[entry].Content));

    pDevice->tx_ring[entry].buffer2 = virt_to_le32desc(skb->data);

    if (q_used_cnt < TX_QUEUE_LEN / 2)
    {	/* Typical path */
	flag = 0x60000000;	/* No interrupt */
    }
    else if(q_used_cnt == TX_QUEUE_LEN / 2)
    {
	flag = 0xe0000000;	/* Tx-done intr. */
    }
    else if (q_used_cnt < TX_QUEUE_LEN)
    {
	flag = 0x60000000;	/* No Tx-done intr. */
    }
    else
    {		
	pDevice->tx_full = 1;
	flag = 0xe0000000;	/* Tx-done intr. */
    }

    if (entry == TX_RING_SIZE - 1)
    {
	flag = 0xe0000000 | DESC_RING_WRAP;
    }

    pDevice->tx_ring[entry].length = cpu_to_le32(Length2);

    pDevice->tx_ring[entry].length = pDevice->tx_ring[entry].length << 12;
    pDevice->tx_ring[entry].length |= flag;

    pDevice->tx_ring[entry].length |= Length1;

    if (entry == TX_RING_SIZE - 1)
    {
	pDevice->tx_ring[entry].length |= 0x02000000;
    }

    pDevice->tx_ring[entry].status = cpu_to_le32(DescOwned);
    pDevice->tx_ring[entry].status |= 0x00a00008;
    pDevice->cur_tx++;

    dev->trans_start = jiffies;
    /* Trigger an immediate transmit demand. */
    outl(0, dev->base_addr + CSR1);

    return 0;
}


/* The interrupt handler does all of the Rx thread work and cleans up
   after the Tx thread. */
static void adm8201_interrupt(int irq, void *dev_instance,
			    struct pt_regs *regs)
{
    struct net_device *dev = (struct net_device *) dev_instance;
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    long ioaddr = dev->base_addr;
    int work_budget = max_interrupt_work;
    DWORD csr5;
   //DisableInterrupt(dev);
    do
    {
		csr5 = inl(ioaddr + CSR5);

		if ((csr5 & (NormalIntr | AbnormalIntr)) == 0)
		    	break;

		outl(csr5 & 0xffffffff, ioaddr + CSR5);

		if( csr5 & BIT_30 )
		{
			pDevice->LM_Device.BeaconCount++;
		}

		if (csr5 & BIT_26 )
		{
			#ifdef lm_debug
				printk("TSFT out of range interrupt happen \n");
			#endif
		}

		if (csr5 & (RxIntr | RxNoBuf))
		{
		    	work_budget -= adm8201_rx_interrupt(dev);
		}

		if (csr5 & (TxNoBuf | TxStopped | TxIntr))
		{
		    	adm8201_tx_interrupt(dev);
		}

		if (pDevice->rx_dead)
		{
	    		adm8201_rx_interrupt(dev);
	    		outl(0, ioaddr + CSR2);	/* Rx poll demand */
	    		pDevice->rx_dead = 0;
		}

		/* Log errors. */
		if (csr5 & AbnormalIntr)
		{	/* Abnormal error summary bit. */


			if (csr5 == 0xffffffff)
				break;

	  	  	if (csr5 & TxLifeTimeOut)
				pDevice->stats.tx_errors++;

	    		if (csr5 & PCIBusError)
			{
				printk("PCI Fatal Bus Error \n");
		    	}

	    		if (csr5 & TxFIFOUnderflow)
			{
				if ((pDevice->csr6 & 0xC000) != 0xC000)
		    			pDevice->csr6 += 0x4000;
					/* Bump up the Tx threshold */
				else
		    			pDevice->csr6 |= 0x00200000;	
					/* Store-n-forward. */

				#ifdef interrupt_debug
		    		printk("%s: Tx threshold increased, new CSR6 %x.\n", dev->name, pDevice->csr6);
				#endif
	    		}

	   		if (csr5 & (TxStopped | TxFIFOUnderflow | PCIBusError))
			{
				/* Restart the transmit process. */
				outl(pDevice->csr6 | RxOn | TxOn, ioaddr + CSR6);
	    		}

			if (csr5 & (RxStopped | RxNoBuf))
			{
				/* Missed a Rx frame or mode change. */
				pDevice->stats.rx_missed_errors += inl(ioaddr + CSR8) & 0xffff;
				
				adm8201_rx_interrupt(dev);
				if (csr5 & RxNoBuf)
		    			pDevice->rx_dead = 1;
				outl(pDevice->csr6 | RxOn | TxOn, ioaddr + CSR6);
	    		}

	    		if (csr5 & TimerInt)
			{
				outl(adm8201_tbl[pDevice->chip_id].valid_intrs, ioaddr + CSR7);
	    		}

	    		if (csr5 & (TPLnkPass | TPLnkFail | 0x08000000))
			{
				 if (pDevice->link_change)
		 	   		(pDevice->link_change) (dev, csr5);
	    		}

	    	outl(0x0800f7ba, ioaddr + CSR5);
		}

		if (--work_budget < 0)
		{

			outl(0x8001ffff, ioaddr + CSR5);

			outl(((~csr5) & 0x0001ebef) | AbnormalIntr | TimerInt,
				ioaddr + CSR7);
			//outl(0x0012, ioaddr + CSR11);
			break;
		}


    } while (1);

    //EnableInterrupt(dev);

    return;
}

static void adm8201_tx_interrupt(struct net_device *dev)
{
    unsigned int dirty_tx;
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
#ifdef packet_debug
	//long ioaddr = dev->base_addr;
	//DWORD Value32;
#endif

    for (dirty_tx = pDevice->dirty_tx; pDevice->cur_tx - dirty_tx > 0; dirty_tx++)
   {
		int entry = dirty_tx % TX_RING_SIZE;
		int status = le32_to_cpu(pDevice->tx_ring[entry].status);

		if (status < 0)
			break;		/* It still has not been Txed */
		/* Check for Rx filter setup frames. */
		if (pDevice->tx_skbuff[entry] == NULL)
			continue;

		if (status & 0x8000)
		{
	    /* There was an major error, log it. */
			pDevice->stats.tx_errors++;

			if (status & BIT_14)
				pDevice->stats.tx_aborted_errors++;  //tx life time out

			if (status & BIT_26)
				pDevice->stats.tx_fifo_errors++; // tx under run error
		}
		else
		{
#ifdef ETHER_STATS
			if (status & 0x0001)
				pDevice->stats.tx_deferred++;
#endif
#if LINUX_VERSION_CODE > 0x20127
			pDevice->stats.tx_bytes += pDevice->tx_skbuff[entry]->len;
#endif
			pDevice->stats.collisions += (status >> 3) & 15;
			pDevice->stats.tx_packets++;
		}


		/* Free the original skb. */
		dev_kfree_skb_irq(pDevice->tx_skbuff[entry]);
		pDevice->tx_skbuff[entry] = 0;

    }

    if (pDevice->tx_full && pDevice->cur_tx - dirty_tx < TX_QUEUE_LEN - 4)
    {
		/* The ring is no longer full, clear tbusy. */
		pDevice->tx_full = 0;
		//netif_resume_tx_queue(dev);
    }

    pDevice->dirty_tx = dirty_tx;
}

static int adm8201_rx_interrupt(struct net_device *dev)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    int entry = pDevice->cur_rx % RX_RING_SIZE;
    int rx_work_limit = pDevice->dirty_rx + RX_RING_SIZE - pDevice->cur_rx;
    int work_done = 0;
    int pkt_len;
    struct sk_buff *skb;
    BYTE MoreFragment = 0;
    BYTE FragmentNum = 0;
    BYTE TypeValue, num ;
    int  len;

    /* If we own the next entry, it is a new packet. Send it up. */
    while (!(pDevice->rx_ring[entry].status & cpu_to_le32(DescOwned)))
    {

		s32 status = le32_to_cpu(pDevice->rx_ring[entry].status);

		pDevice->rx_ring[entry].length = cpu_to_le32(pDevice->RDES1[entry]);

#ifdef adm8201_debug
		if (adm8201_debug > 5) {
			printk(KERN_DEBUG "%s: In adm8201_rx_interrupt(), entry %d %8.8x.\n",
			dev->name, entry, status);
		}
#endif

		if (--rx_work_limit < 0) {
			printk("rx_work_limit < 0 \n");
			break;
		}

		pkt_len = (status & 0x7ff);

		if ((status & 0x01b80000) && (pDevice->LM_Device.RevisionStep < REVISION_BA))
		{
			#ifdef adm8201_debug
				printk("Rx error packet\n");
			#endif
			if (status & BIT_20 ) pDevice->stats.rx_crc_errors++;
			if (status & BIT_21 ) pDevice->stats.rx_fifo_errors++;

		}
		else
		{
			skb_put(skb = pDevice->rx_skbuff[entry], pkt_len);

			TypeValue = (skb->data[0] & 0x0f);
			switch (TypeValue) {
			case MANAGMENT_FRAME:
				#ifdef packet_debug
				  printk("cf = %02x   %02x .....\n",skb->data[0],skb->data[1]);
				#endif
				/*Jeffrey: for link quality*/
				if(skb->data[0]==0x80
				&&skb->data[16]==
				pDevice->STA_Module.MDL_AP_Info[pDevice->STA_Module.MDL_AP_Inuse_Index].BSSID[0]
				
		  		&&skb->data[17]==
				pDevice->STA_Module.MDL_AP_Info[pDevice->STA_Module.MDL_AP_Inuse_Index].BSSID[1]
				
				&&skb->data[18]==
				pDevice->STA_Module.MDL_AP_Info[pDevice->STA_Module.MDL_AP_Inuse_Index].BSSID[2]
				
				&&skb->data[19]==
				pDevice->STA_Module.MDL_AP_Info[pDevice->STA_Module.MDL_AP_Inuse_Index].BSSID[3]
				
				&&skb->data[20]==
				pDevice->STA_Module.MDL_AP_Info[pDevice->STA_Module.MDL_AP_Inuse_Index].BSSID[4]
				
				&&skb->data[21]==
				pDevice->STA_Module.MDL_AP_Info[pDevice->STA_Module.MDL_AP_Inuse_Index].BSSID[5]
				)	
				{
					pDevice->beaconcount++ ;
				} 
				TypeValue = (skb->data[0] & 0xf0);
				UM_MmIndicate(&(pDevice->STA_Module), TypeValue, skb->data, pkt_len);
				dev_kfree_skb(skb);
				break;
			case CONTROL_FRAME:
				dev_kfree_skb(skb);
				break;

			case DATA_FRAME:
				FragmentNum = skb->data[22] & 0x0f;
				MoreFragment = skb->data[1] & 0x04;

				if (IS_BROADCAST_MAC( skb->data+16 ))
				{
					if (skb->data[10] == pDevice->STA_Module.MDL_STA_MAC[0] &&
						skb->data[11] == pDevice->STA_Module.MDL_STA_MAC[1] &&
						skb->data[12] == pDevice->STA_Module.MDL_STA_MAC[2] &&
						skb->data[13] == pDevice->STA_Module.MDL_STA_MAC[3] &&
						skb->data[14] == pDevice->STA_Module.MDL_STA_MAC[4] &&
						skb->data[15] == pDevice->STA_Module.MDL_STA_MAC[5])
					{
						printk(" receive myselt packet \n");
						dev_kfree_skb(skb);

					}
				}
				else
				{
					// Rx Software wep
					if ((pDevice->STA_Module.MDL_WEP_Enable != 0)
						&& ((skb->data[1] & BIT_6) == BIT_6))
					{
						if(skb->data[27] ==0x00)
						{
				pDevice->STA_Module.MDL_WEP_Key_Index = 0;	
						}else if(skb->data[27] ==0x40)	
						{
				pDevice->STA_Module.MDL_WEP_Key_Index = 1;	
						}else if(skb->data[27] ==0x80)
						{	
				pDevice->STA_Module.MDL_WEP_Key_Index = 2;	
						}else if(skb->data[27] ==0xc0)
						{
				pDevice->STA_Module.MDL_WEP_Key_Index = 3;	
						}	

					if(pDevice->LM_Device.RevisionStep< 
							REVISION_BA)
					{
						UM_Software_WEP(&(pDevice->STA_Module), skb->data, &pkt_len);	
					}

					}
					// Rx Defragment
					if ( ( MoreFragment != 0 ) || (FragmentNum != 0))
					{
						DefragmentPacketAssemble(&(pDevice->LM_Device), skb->data, pkt_len);
						if ( pDevice->LM_Device.CompleteQueueNum != 0 )
						{
							num = pDevice->LM_Device.CompleteQueueNum-1;
							len=pDevice->LM_Device.RxDef[num].data_len;

							skb->data = pDevice->LM_Device.RxDef[num].data;
							pDevice->LM_Device.RxDef[num].data_len=0;

							if (pDevice->LM_Device.NumofQueue != 0)
								pDevice->LM_Device.NumofQueue--;

							pkt_len=len;
							pDevice->LM_Device.CompleteQueueNum=0;

							#ifdef	defragment_debug
								for (i=0;i<pkt_len;i++)
								{
									if (( i!=0 ) && ( i%8 == 0) )
										printk("\n");
									printk("%02x ",skb->data[i]);

								}
								printk("\n");
							#endif
						}
					}

					skb->data = (unsigned char *)UM_11to3(&(pDevice->STA_Module), skb->data, &pkt_len);

					skb->len = pkt_len;

					// Indicate to Upper layer
					skb->protocol = eth_type_trans(skb, dev);
					netif_rx(skb);
					dev->last_rx = jiffies;
				}

				break;

			default:
				break;
			}
		}	//if ( status & BIT_30 )
		pDevice->rx_skbuff[entry] = NULL;

		//Jeffrey06232003
		pDevice->RxPacketCountPer2S++;
		pDevice->stats.rx_packets++;

	#if LINUX_VERSION_CODE > 0x20127
		pDevice->stats.rx_bytes += pkt_len;
	#endif
		entry = (++pDevice->cur_rx) % RX_RING_SIZE;
    }

    /* Refill the Rx ring buffers. */
    for (; pDevice->cur_rx - pDevice->dirty_rx > 0; pDevice->dirty_rx++) {
	entry = pDevice->dirty_rx % RX_RING_SIZE;
	if (pDevice->rx_skbuff[entry] == NULL) {
	    struct sk_buff *skb;
	    //alloc memory for Rx desc
	    skb = pDevice->rx_skbuff[entry] = dev_alloc_skb(PKT_BUF_SZ);
	    if (skb == NULL)
	    {
		printk("The system can not allocate any memory\n");    
		break;
	    }
	    skb->dev = dev;	/* Mark as being used by this device. */
	    pDevice->rx_ring[entry].buffer1 = virt_to_le32desc(skb->tail);
	    work_done++;
	}
	pDevice->rx_ring[entry].status = cpu_to_le32(DescOwned);
    }
    return work_done;
}

//*********************************************************************

void empty_rings(struct net_device *dev)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    int i;

    /* Free all the skbuffs in the Rx queue. */
    for (i = 0; i < RX_RING_SIZE; i++) {
	struct sk_buff *skb = pDevice->rx_skbuff[i];
	pDevice->rx_skbuff[i] = 0;
	pDevice->rx_ring[i].status = 0;	/* Not owned by adm8201 chip. */
	pDevice->rx_ring[i].length = 0;
	pDevice->rx_ring[i].buffer1 = 0xBADF00D0;/* An invalid address. */
	if (skb) {
	    dev_kfree_skb(skb);
	}
    }
    for (i = 0; i < TX_RING_SIZE; i++) {
	if (pDevice->tx_skbuff[i])
	    dev_kfree_skb(pDevice->tx_skbuff[i]);
	pDevice->tx_skbuff[i] = 0;
    }

}

static void __devexit adm8201_remove(struct pci_dev *pdev)
{
    struct net_device *dev = pci_get_drvdata (pdev);
    
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;

    printk("ADM8211 remove\n");
    if (!dev)
        return;

    unregister_netdev (dev);
    
    if (pDevice->priv_addr)
          kfree(pDevice->priv_addr);
    pci_release_regions (pdev);
    kfree (dev);
    pci_set_drvdata (pdev, NULL);
    
}

static int adm8201_close(struct net_device *dev)
{

    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    long ioaddr = dev->base_addr;

#ifdef ioctl_debug
    printk("adm8201 close \n");
#endif
    outl(0, ioaddr + CSR6);
    DisableInterrupt(dev);
    pDevice->LM_Device.UmInitModule_Flag=1;
    /*Jeffrey 11012002*/
    pDevice->LM_Device.Start_Tx = 0 ;
    netif_stop_queue(dev);
    
 
   /*Jeffrey 11012002*/
    LM_Del_Timer();
    del_timer(&pDevice->timer);
    free_irq(dev->irq, dev);
    return 0;
}


static struct net_device_stats *adm8201_get_stats(struct net_device *dev)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    long ioaddr = dev->base_addr;
    int csr8 = inl(ioaddr + CSR8);

    if (netif_running(dev) && csr8 != 0xffffffff)
    	   pDevice->stats.rx_missed_errors += (u16) csr8;

    return &pDevice->stats;
}



static int adm8201_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;

    struct iwreq *wrq = (struct iwreq *)rq;
    struct mii_ioctl_data *data = (struct mii_ioctl_data *) & rq->ifr_data;

    int err = 0;
    struct iw_range range;
    int i=0,k=0;
    BYTE	query_buf[128];
    int		query_len=0;
    DWORD	value32;
    BYTE	AP_Num;
    int		index;
   

	if (! netif_device_present(dev))
		return -ENODEV;

	switch (cmd)
	{
		case SIOCGIWNAME:	/* get name == wireless protocol */
			strcpy(wrq->u.name, "IEEE 802.11-DS");
			break;

		case SIOCGIWNWID:
			err = -EOPNOTSUPP;
			break;

		case SIOCGIWAP:		/* get access point MAC addresses */
			wrq->u.ap_addr.sa_family = ARPHRD_ETHER;
			if( (pDevice->STA_Module.MDL_Assoc_Complete == 1) ||
				( pDevice->STA_Module.MDL_Current_Action == ALL_COMPLETE ) )
			{
				UM_Query_Information(&(pDevice->STA_Module), OID_802_11_BSSID, wrq->u.ap_addr.sa_data, &query_len);
			}
			else
			{
				for (i=0; i<6 ;i++ )
				{
					wrq->u.ap_addr.sa_data[i]=0x00;
				}
			}
			break;

		case SIOCGIWRANGE:		/* Get range of parameters */
			err = verify_area(VERIFY_WRITE, wrq->u.data.pointer, sizeof(range));
			if (err)
				return err;
			wrq->u.data.length = sizeof(range);
			//mode does not be filled  yet
			memset(&range, 0, sizeof(range));

#if WIRELESS_EXT > 10
			range.we_version_compiled = WIRELESS_EXT;
			range.we_version_source = 12;
#endif /* WIRELESS_EXT > 10 */

			range.min_nwid = range.max_nwid = 0; /* We don't use nwids */
			range.num_channels = NUM_CHANNELS;
			k = 0;
			for (i = 0; i < NUM_CHANNELS; i++) {
				range.freq[k].i = i + 1;
				range.freq[k].m = channel_frequency[i];
				range.freq[k].e = 1;
				k++;
				if (k >= IW_MAX_FREQUENCIES)
					break;
			}
			range.num_frequency = k;
			range.sensitivity = 65535;
			//if (mode == IW_MODE_ADHOC)
			  range.max_qual.qual = 200;//10
			  range.max_qual.level = 0x100 - 120;     /* -120 dBm */
			  range.max_qual.noise = 0;
			  range.sensitivity = 65535;

#if WIRELESS_EXT > 11
			 range.avg_qual.qual = 80;//6
			 range.avg_qual.level = 176;     /* -80 dBm */
			 range.avg_qual.noise = 0;
#endif /* WIRELESS_EXT > 11 */

			for(i = 0 ; i < 4 ; i++)
			{
				range.bitrate[i] = ((pDevice->STA_Module.MDL_Support_Rate[i+1])&0x7f)*500000;
			}
			range.num_bitrates = 4;

			range.min_rts = 0;
			range.max_rts = 2346;
			range.min_frag = 256;
			range.max_frag = 2346;
			/*Wep*/
			 if (pDevice->STA_Module.MDL_WEP_Enable!=0) {
				range.max_encoding_tokens = 4;
				range.encoding_size[0] = 5;
				range.encoding_size[1] = 13;
				range.num_encoding_sizes = 2;
			}else
			{
				range.num_encoding_sizes = 0;
				range.max_encoding_tokens = 0;
			}

			range.min_pmp = 0;
			range.max_pmp = 65535000;
			range.min_pmt = 0;
			range.max_pmt = 65535 * 1000;   /* ??? */
			range.pmp_flags = IW_POWER_PERIOD;
			range.pmt_flags = IW_POWER_TIMEOUT;
			range.pm_capa = IW_POWER_PERIOD | IW_POWER_TIMEOUT | IW_POWER_UNICAST_R;

			range.num_txpower = 1;
			range.txpower[0] = 15; /* 15dBm */
			range.txpower_capa = IW_TXPOW_DBM;

#if WIRELESS_EXT > 10
				range.retry_capa = IW_RETRY_LIMIT | IW_RETRY_LIFETIME;
				range.retry_flags = IW_RETRY_LIMIT;
				range.r_time_flags = IW_RETRY_LIFETIME;
				range.min_retry = 1;
				range.max_retry = 1;
				range.min_r_time = 1024;
				range.max_r_time = 65535 * 1024;
#endif /* WIRELESS_EXT > 10 */

			if (copy_to_user(wrq->u.data.pointer, &range, sizeof(range)))
							return -EFAULT;
			break;

		case SIOCSIWMODE:		/* set operation mode */
			#ifdef  ioctl_debug
				printk("Set operation mode %x\n",wrq->u.mode);
			#endif
			switch (wrq->u.mode)
			{
				case IW_MODE_ADHOC:
					#ifdef ioctl_debug
						printk("this is ADHOC mode \n");
					#endif
					pDevice->LM_Device.iw_mode = Ndis802_11IBSS;
					pDevice->STA_Module.MDL_Operation_Mode = Ndis802_11IBSS;	//set adhoc mode
					break;

				case IW_MODE_INFRA:
					pDevice->LM_Device.iw_mode = Ndis802_11Infrastructure;
					pDevice->STA_Module.MDL_Operation_Mode = Ndis802_11Infrastructure;	//set infrastructure mode
					#ifdef ioctl_debug
						printk("this is infrastructure mode \n");
					#endif
					break;

				default:
					#ifdef ioctl_debug
						printk(" not support mode \n");
					#endif
					err = -EINVAL;
					break;
			}
			break;

		case SIOCGIWMODE:		/* get operation mode */
			switch(pDevice->LM_Device.iw_mode)
			{
				case Ndis802_11Infrastructure:
					wrq->u.mode = IW_MODE_INFRA;
					break;

				case Ndis802_11IBSS:
					wrq->u.mode = IW_MODE_ADHOC;
					break;

				default:
					break;
			}
			break;

		case SIOCSIWENCODE:		/* set encoding token & mode */

			#ifdef ioctl_debug
				printk("Set wep key\n");
				printk("before : wrq->u.encoding.length =%x \n"
						,wrq->u.encoding.length);
				printk("wrq->u.encoding.flags =%x \n"
						,wrq->u.encoding.flags);
			#endif
			memcpy(query_buf, wrq->u.encoding.pointer
					,wrq->u.encoding.length);

			if ( (( wrq->u.encoding.length == 5 )|| 
					(wrq->u.encoding.length== 13 ))
			      && (wrq->u.encoding.flags < 5) )
			{
				memcpy(query_buf, wrq->u.encoding.pointer
						,wrq->u.encoding.length);
				#ifdef ioctl_debug
					for (i=0;i<wrq->u.encoding.length;i++)
					printk(" %02x",query_buf[i]);
				printk("\n");
				#endif

			#ifdef  ioctl_debug
				printk("##########\nindex is %x\n",
					pDevice->STA_Module.MDL_WEP_Key_Index);
			#endif
				if ( wrq->u.encoding.length == 5)
			   	      pDevice->STA_Module.MDL_WEP_Key_Length=5;
				else
				      pDevice->STA_Module.MDL_WEP_Key_Length=13;

				for ( i=0;i<wrq->u.encoding.length;i++)
				{
//if key do no assign 
				if (wrq->u.encoding.flags == 0)
				    	pDevice->STA_Module.MDL_WEP_Key[0][i]=query_buf[i];
				else   
				pDevice->STA_Module.MDL_WEP_Key[wrq->u.encoding.flags-1][i]=query_buf[i];
				}
				UM_Set_Information(&(pDevice->STA_Module), OID_802_11_ADD_WEP, query_buf, &query_len);
				#ifdef ioctl_debug
					printk("after set key vlaue\n");
					for (i=0;i<wrq->u.encoding.length;i++)
						printk(" %02x",
						pDevice->STA_Module.
						MDL_WEP_Key[pDevice->STA_Module.
						MDL_WEP_Key_Index][i]);
					printk("\n\n");
				#endif

			}
			
			if ( ( wrq->u.encoding.length == 5 )|| 
			     ( wrq->u.encoding.length == 13 ) ||
			     ( wrq->u.encoding.length == 0 ) )
			{
				if(wrq->u.encoding.flags & IW_ENCODE_DISABLED)
				{
					pDevice->LM_Device.authType = AUTH_OPEN;
					UM_Set_Information((&pDevice->STA_Module), OID_802_11_REMOVE_WEP , query_buf, &query_len );
				}

				if(wrq->u.encoding.flags & IW_ENCODE_RESTRICTED)
					pDevice->LM_Device.authType = 
						AUTH_SHAREDKEY;	// Only Both
				
				if((wrq->u.encoding.flags & IW_ENCODE_OPEN) 
					|| (wrq->u.encoding.flags == 0 )
					|| (wrq->u.encoding.flags == 1 ))
					pDevice->LM_Device.authType = 
						AUTH_ENCRYPT;	// Only Wep
			}
			/*set which index to use*/
			if (wrq->u.encoding.length == 0)
			{
				if (wrq->u.encoding.flags < 4)
			  	      pDevice->STA_Module.MDL_WEP_Key_Index=
					(wrq->u.encoding.flags-1);
					
			}	

			break;

		case SIOCGIWENCODE:		/* get encoding token & mode */
			#ifdef ioctl_debug
			printk("get wrq->u.encoding.length is %x\n",
			       wrq->u.encoding.length);
			printk("get wrq->u.encoding.flags is %x\n",
				wrq->u.encoding.flags);
			#endif
			if ( pDevice->LM_Device.authType == AUTH_OPEN)
			{
				// disable wep
				wrq->u.encoding.length=0;
				wrq->u.encoding.flags|=IW_ENCODE_DISABLED;
				wrq->u.encoding.flags&=0x00;
			}
			else
			{
				
			if (wrq->u.encoding.flags == 0 )
				index = pDevice->STA_Module.MDL_WEP_Key_Index;
			else	
				index = wrq->u.encoding.flags - 1;
			
			memcpy(query_buf,
			pDevice->STA_Module.MDL_WEP_Key[index],
			pDevice->STA_Module.MDL_WEP_Key_Length);
				

			copy_to_user(wrq->u.encoding.pointer, query_buf
			, pDevice->STA_Module.MDL_WEP_Key_Length);
				
			wrq->u.encoding.length=
				pDevice->STA_Module.MDL_WEP_Key_Length;

			if ( pDevice->LM_Device.authType == AUTH_SHAREDKEY )
				wrq->u.encoding.flags= 
				IW_ENCODE_RESTRICTED | IW_ENCODE_NOKEY;
			else
				wrq->u.encoding.flags=IW_ENCODE_OPEN;  // only wep
			/*Jeffrey : get current transmit key index*/
			wrq->u.encoding.flags |= 
				(pDevice->STA_Module.MDL_WEP_Key_Index+0x01);

			}
			break;

		case SIOCSIWESSID:		/* set ESSID (network name) */
			query_len = wrq->u.essid.length;
			query_len--;
			#ifdef ioctl_debug
			printk("Set SSID ..\n");
			printk("set ssid length =%d \n",wrq->u.essid.length);
			printk("set ssid flags =%x \n",wrq->u.essid.flags);
			#endif

			memcpy(query_buf,wrq->u.essid.pointer,query_len);
			#ifdef ioctl_debug
				for (i=0;i<query_len;i++)
					printk(" %x ",query_buf[i]);
				printk("\n");
			#endif
			if(query_len == 0)
			{
				query_len = 3;
				query_buf[0]='a';
				query_buf[1]='n';
				query_buf[2]='y';
			}
			UM_Set_Information(&(pDevice->STA_Module), OID_802_11_SSID, query_buf, &query_len);

			if( ( pDevice->LM_Device.UmInitModule_Flag == 0 ) && (pDevice->LM_Device.Start_ADM8211 == 1) )
			{
				//only for iwcofnig use
				#ifdef ioctl_debug
					printk("ioctl set ssid call UM_Init_Module \n");
				#endif
				UM_Init_Module(&pDevice->STA_Module);
			}
			break;

		case SIOCGIWESSID:		/* get ESSID */
			if( (pDevice->STA_Module.MDL_Assoc_Complete == 1) ||
				( pDevice->STA_Module.MDL_Current_Action == ALL_COMPLETE ) )
			{
				UM_Query_Information(&(pDevice->STA_Module), OID_802_11_SSID, query_buf, &query_len);
				copy_to_user(wrq->u.essid.pointer, query_buf, query_len);
				wrq->u.essid.length=query_len;
			}
			else
			{
				wrq->u.essid.length=0;
			}
			wrq->u.essid.flags=1;
			break;

		case SIOCSIWNICKN:		/* set node name/nickname */
			#ifdef ioctl_debug
				printk("Set nick name\n");
			#endif
			if (wrq->u.data.length > IW_ESSID_MAX_SIZE)
				return -E2BIG;

			memset(query_buf, 0, sizeof(query_buf));

	  	       if (copy_from_user(query_buf, wrq->u.data.pointer, wrq->u.data.length))
		                return -EFAULT;
		
			query_buf[wrq->u.data.length] = '\0';

			memcpy(pDevice->LM_Device.NicName, query_buf, IW_ESSID_MAX_SIZE+1);
			break;

		case SIOCGIWNICKN:		/* get node name/nickname */
			if (pDevice->LM_Device.NicName[0] != 0 )
			{
				memcpy(query_buf, pDevice->LM_Device.NicName, IW_ESSID_MAX_SIZE+1);
				wrq->u.data.length = strlen(query_buf)+1;

				if (copy_to_user(wrq->u.data.pointer, query_buf, wrq->u.data.length))
						return -EFAULT;
			}
			break;

		case SIOCSIWFREQ:/* set channel/frequency (Hz) */
			#ifdef ioctl_debug
				printk("set channel/freq value =%d , type=%x\n",wrq->u.freq.m ,wrq->u.freq.e);
			#endif
			if ( wrq->u.freq.e == 1 )
			{
				for (i=0;i<14;i++)
				{
					if ( wrq->u.freq.m == channel_frequency[i] )
					{
						wrq->u.freq.m=i+1;
						pDevice->LM_Device.freq_channel=1;
						break;
					}
				}
			}
			else
			{
				pDevice->LM_Device.freq_channel=0;
			}
			if (i==14)
				return -EFAULT;
			query_buf[0]=wrq->u.freq.m;
			if (query_buf[0] > 14 )
				return -EFAULT;

			if ( query_buf[0] != pDevice->STA_Module.MDL_Channel )
			{
				UM_Set_Information(&(pDevice->STA_Module), OID_802_11_CHANNEL, query_buf, &query_len);
			}
			break;

		case SIOCGIWFREQ:/* get channel/frequency (Hz) */

			if( (pDevice->STA_Module.MDL_Assoc_Complete == 1) ||
				( pDevice->STA_Module.MDL_Current_Action == ALL_COMPLETE ) )
			{
				UM_Query_Information(&(pDevice->STA_Module), OID_802_11_CHANNEL, query_buf, &query_len);

				if ( pDevice->LM_Device.freq_channel == 0 )
				{
					wrq->u.freq.m = query_buf[0];
					wrq->u.freq.e = 0;    // 0: chnnel  1:freq
				}
				else
				{
					wrq->u.freq.m = channel_frequency[query_buf[0]-1];
					wrq->u.freq.e = 1;    // 0: chnnel  1:freq
				}
				wrq->u.freq.m = channel_frequency[query_buf[0]-1];

			}
			else
			{
				wrq->u.freq.m = 0 ;
			}
			wrq->u.freq.e = 1;    // 0: chnnel  1:freq
			break;

		case SIOCSIWSENS:/* set sensitivity (dBm) */
			wrq->u.sens.fixed = 0;
			wrq->u.sens.disabled = 1;

			break;

		case SIOCGIWSENS:/* get sensitivity (dBm) */
			wrq->u.sens.value = 0;
			wrq->u.sens.fixed = 0;
			wrq->u.sens.disabled = 1 ;
			break;

		case SIOCSIWRTS:/* set RTS/CTS threshold (bytes) */
			value32 = wrq->u.rts.value;
			if ( value32 == -1 )
			{
				value32=2346;
			}
			query_buf[0]=(BYTE)value32&0xff;
			query_buf[1]=(BYTE)((value32>>8)&0xff);
			UM_Set_Information(&(pDevice->STA_Module), OID_802_11_RTS_THRESHOLD, query_buf, &query_len);
			#ifdef ioctl_debug
				printk("After Set RTS threshold is %x\n",pDevice->STA_Module.MDL_RTS_Thd);
			#endif
			break;

		case SIOCGIWRTS:/* get RTS/CTS threshold (bytes) */
			UM_Query_Information(&(pDevice->STA_Module), OID_802_11_RTS_THRESHOLD, query_buf, &query_len);
			wrq->u.rts.value=(__s32)((*query_buf)+(*(query_buf+1)<<8));
			wrq->u.rts.fixed = 1;
			wrq->u.rts.disabled = 0;
			break;

		case SIOCSIWFRAG:/* set fragmentation thr (bytes) */
			value32 = wrq->u.frag.value;
			if ( value32 == -1 )
			{
				value32=2346;
			}
			query_buf[0]=(BYTE)value32&0xff;
			query_buf[1]=(BYTE)((value32>>8)&0xff);
			UM_Set_Information(&(pDevice->STA_Module), OID_802_11_FRAGMENTATION_THRESHOLD, query_buf, &query_len);
			#ifdef ioctl_debug
				printk("After Set fragmentation thr is %i\n",pDevice->STA_Module.MDL_Frag_Thd);

			#endif

			break;

		case SIOCGIWFRAG:/* get fragmentation thr (bytes) */
			UM_Query_Information(&(pDevice->STA_Module), OID_802_11_FRAGMENTATION_THRESHOLD, query_buf, &query_len);
			wrq->u.frag.value=(__s32)((*query_buf)+(*(query_buf+1)<<8));
			wrq->u.frag.fixed = 1;
			wrq->u.frag.disabled = 0;
			break;

		case SIOCSIWRATE:/*set default bit rate (bps)*/
			value32 = wrq->u.bitrate.value/100000;
			query_buf[0]=(BYTE)value32;
			query_len=1;
			UM_Set_Information(&(pDevice->STA_Module), OID_802_11_SUPPORTED_RATES , query_buf, &query_len);
			#ifdef ioctl_debug
				printk("Set bit rate default %d\n",query_buf[0]);
				printk("After Set bit rate value is %d\n",pDevice->STA_Module.MDL_Data_Rate);
			#endif
			break;

		case SIOCGIWRATE:/* get default bit rate (bps) */

			UM_Query_Information(&(pDevice->STA_Module), OID_802_11_SUPPORTED_RATES , query_buf, &query_len);
			if (query_buf[0] == 0 )
			{
				query_buf[0]=pDevice->STA_Module.MDL_Max_Data_Rate;
				pDevice->STA_Module.MDL_Data_Rate = pDevice->STA_Module.MDL_Max_Data_Rate;
			}

			wrq->u.bitrate.value = query_buf[0]*100000;
			wrq->u.bitrate.disabled = 0;
			break;

/***** Power saving stuff(power management, unicast and multicast) *****/
		case SIOCSIWPOWER:/* set Power Management settings */
			//wrq->u.power.fixed = 1;
			wrq->u.power.flags=0;
			wrq->u.power.value=0;
			wrq->u.power.disabled = 1;

			break;

		case SIOCGIWPOWER:/* get Power Management settings */
			//wrq->u.power.fixed = 1;
			wrq->u.power.flags=0;
			wrq->u.power.value=0;
			wrq->u.power.disabled = 1;
			break;
/* *************     end of Power saving	***********	*/
		case SIOCSIWTXPOW:/* set transmit power (dBm) */

			#ifdef ioctl_debug
			printk("Set TxPower value =%d fixed=%x disabled=%x flags=%x \n",wrq->u.txpower.value,
				wrq->u.txpower.fixed, wrq->u.txpower.disabled, wrq->u.txpower.flags);
			#endif

			if( wrq->u.txpower.disabled == 1)
				pDevice->LM_Device.TxPowerFlag=Disable;
			else
			{
				if (( wrq->u.txpower.value >= 0x00 ) && ( wrq->u.txpower.value <= 0x3f ) )
				{
					if( wrq->u.txpower.value == -1 )
					{
						wrq->u.txpower.value=0x2f;
					}
					pDevice->LM_Device.TxPowerFlag=Enable;
					netif_stop_queue(dev);
					query_buf[0]=wrq->u.txpower.value;
					UM_Set_Information(&(pDevice->STA_Module), OID_802_11_TX_POWER_LEVEL , query_buf, &query_len);
					netif_start_queue(dev);

				}
			}

			break;
		case SIOCGIWTXPOW:/* get transmit power (dBm) */
			/*we should know current channel*/
			UM_Query_Information(&(pDevice->STA_Module), OID_802_11_TX_POWER_LEVEL , query_buf, &query_len);
			wrq->u.txpower.value=query_buf[0];
			wrq->u.txpower.fixed = 1;
			if (pDevice->LM_Device.TxPowerFlag == Disable )
				wrq->u.txpower.disabled = 1;
			else
				wrq->u.txpower.disabled = 0;
			wrq->u.txpower.flags = IW_TXPOW_DBM;

			break;

#if WIRELESS_EXT > 10
		case SIOCSIWRETRY:
			#ifdef ioctl_debug
			printk("retry  value =%d fixed=%x disabled=%x flags=%x \n",wrq->u.retry.value, wrq->u.retry.fixed , wrq->u.retry.disabled, wrq->u.retry.flags);
			#endif
			if( wrq->u.retry.flags == IW_RETRY_LIMIT )
			{
				pDevice->STA_Module.MDL_Retry_Count=(BYTE)wrq->u.retry.value;
			}
			break;

		case SIOCGIWRETRY:
			wrq->u.retry.value=pDevice->STA_Module.MDL_Retry_Count;
			wrq->u.retry.flags=IW_RETRY_LIMIT;
			break;
#endif /* WIRELESS_EXT > 10 */

		case SIOCSIWSPY:/* set spy addresses */
			err = -EOPNOTSUPP;
			break;

		case SIOCGIWSPY:/* get spy info (quality of link) */
			err = -EOPNOTSUPP;
			break;

		case SIOCGIWPRIV:/* get private info */
			err = -EOPNOTSUPP;
			break;

		case SIOCGIWAPLIST:
			if (wrq->u.data.pointer)
			{
				struct sockaddr s[IW_MAX_AP];

				if ( pDevice->STA_Module.MDL_AP_Info_Num > IW_MAX_AP )
					AP_Num = IW_MAX_AP;
				else
					AP_Num = pDevice->STA_Module.MDL_AP_Info_Num;
				for	(i = 0; i < AP_Num ; i++)
				{
					memcpy(s[i].sa_data, pDevice->STA_Module.MDL_AP_Info[i].BSSID, 6);
					s[i].sa_family = ARPHRD_ETHER;
				}

				wrq->u.data.length = pDevice->STA_Module.MDL_AP_Info_Num;
				copy_to_user(wrq->u.data.pointer, &s,sizeof(struct sockaddr)*AP_Num);

			}
			break;

		/* additional ioctls (sockios.h) */
		case SIOCGMIIPHY:
		case SIOCDEVPRIVATE:
			data->phy_id = 1;
			/* fall through */

		case SIOCGMIIREG:	
		case SIOCDEVPRIVATE+1:
			data->val_out = 0x0004; /* link ok */
			break;

		case SIOCETHTOOL:
			/* could call netdev_ethtool_ioctl(dev, (void *) rq->ifr_data);  ?? */
			err = -EOPNOTSUPP;
			break;

		default:
			printk(KERN_INFO "8211: unhandled ioctl = %08x\n",cmd);
			err = -EOPNOTSUPP;
			break;
	}

	return err;
}


static struct pci_driver adm8211_driver = {
	name:		DRV_NAME,
	id_table:	pci_id_tbl,
	probe:		adm8211_init_one,
	remove:		__devexit_p(adm8201_remove),
};


static int __init adm8211_init (void)
{

	/* probe for and init boards */
	return pci_module_init (&adm8211_driver);
}


static void __exit adm8211_cleanup (void)
{
	pci_unregister_driver (&adm8211_driver);
}


module_init(adm8211_init);
module_exit(adm8211_cleanup);

/*****************************************************************************/

void adm8201_Init(struct net_device *dev)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    long ioaddr = dev->base_addr;
    int next_tick = 3 * HZ;

    outl(0, ioaddr + CSR6);	/* for soft rest */
    ChipReset(&(pDevice->LM_Device));

    adm8201_init_ring(dev );

    Hardware_Init(&(pDevice->LM_Device));

#ifdef adm8201_debug
    if (adm8201_debug > 1)
    {
	printk(KERN_DEBUG "%s: adm8201_Init() irq %d.\n", dev->name,
	       dev->irq);
    }
#endif

    outl(virt_to_bus(pDevice->rx_ring), ioaddr + CSR3);
    outl(virt_to_bus(pDevice->tx_ring), ioaddr + CSR4);


    /* Start the Tx to process setup frame. */

    /* Enable interrupts by setting the interrupt mask. */
    outl(adm8201_tbl[pDevice->chip_id].valid_intrs, ioaddr + CSR5);

    outl(0, ioaddr + CSR2);	/* Rx poll demand */


#ifdef adm8201_debug
    if (adm8201_debug > 2)
    {
	printk(KERN_DEBUG
	       "%s: Done adm8201_Init(), CSR0 %8.8x, CSR5 %8.8x CSR6 "
	       "%8.8x.\n", dev->name, (int) inl(ioaddr + CSR0),
	       (int) inl(ioaddr + CSR5), (int) inl(ioaddr + CSR6));
    }
#endif

    init_timer(&pDevice->timer);
    pDevice->timer.expires = jiffies + next_tick;
    pDevice->timer.data = (unsigned long) dev;
    pDevice->timer.function = adm8201_tbl[pDevice->chip_id].media_timer;
    add_timer(&pDevice->timer);

    netif_stop_queue(dev);
}

void DisableInterrupt(struct net_device *dev)
{

    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    long ioaddr = dev->base_addr;
    pDevice->LM_Device.IntMask2 = 0;
    outl(0, ioaddr + CSR7);

}			

void EnableInterrupt(struct net_device *dev)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    long ioaddr = dev->base_addr;

    pDevice->LM_Device.IntMask2 = pDevice->LM_Device.IntMaskBackUp;
    outl(pDevice->LM_Device.IntMask2, ioaddr + CSR7);


}				/* LM_EnableInterrupt */

/************************************************************************/
/*			Timer Function 					*/
/* 		it should be done init_timer before call this function	*/
/*		del_timer(&LM_timer_list) in adm8201-close 		*/
/************************************************************************/

int LOCAL_Timer_func(STA_MDL * pSTA_Module)
{
    pSTA_Module->MDL_Time_Function((unsigned long *)pSTA_Module);
    LM_timer_list.function = (void *)LOCAL_Timer_func;
    LM_timer_list.expires = jiffies + pSTA_Module->MDL_Time;

    if (pSTA_Module->MDL_Current_Action != 0x10) {
	add_timer(&LM_timer_list);
    }

	return 0;
}

void LM_Set_Timer(STA_MDL * pSTA_Module)
{

    LM_timer_list.data = (unsigned long) pSTA_Module;
    LM_timer_list.function = (void *)LOCAL_Timer_func;
    LM_timer_list.expires = jiffies + pSTA_Module->MDL_Time;
    add_timer(&LM_timer_list);
}
void LM_Del_Timer(void)
{
    del_timer(&LM_timer_list);
}


/************************************************************************/
/*			Memory Function					*/
/************************************************************************/
UINT8 *LM_Get_Memory(int Mem_len)
{

    void *ptr = kmalloc((size_t) Mem_len, GFP_KERNEL);
    return ptr;
}


/************************************************************************/
/*			 Send packet Function				*/
/************************************************************************/
void LM_Send_Packet(STA_MDL * pSTA_Module)
{
    struct net_device *dev1 = (struct net_device *) pSTA_Module->MDL_pDrvObj;
    struct adm8201_private *pDevice = (struct adm8201_private *) dev1->priv;
    int entry, q_used_cnt;
    u32 flag;
    s32 Status;

    struct sk_buff *skb = dev_alloc_skb(512);

    entry = pDevice->cur_tx % TX_RING_SIZE;
    q_used_cnt = pDevice->cur_tx - pDevice->dirty_tx;
    pDevice->tx_skbuff[entry] = skb;

    memcpy(skb_put(skb, pSTA_Module->MDL_TX_Len), pSTA_Module->MDL_TX_Buf,
	   pSTA_Module->MDL_TX_Len);
    UM_Free_Memory(pSTA_Module);

    pDevice->tx_ring[entry].buffer1 = virt_to_le32desc(skb->data);
    if (q_used_cnt < TX_QUEUE_LEN / 2)
    {	/* Typical path */
		flag = 0x60000000;	/* No interrupt */
    }
    else if (q_used_cnt == TX_QUEUE_LEN / 2)
    {
		flag = 0xe0000000;	/* Tx-done intr. */
    }
    else if (q_used_cnt < TX_QUEUE_LEN)
    {
		flag = 0x60000000;	/* No Tx-done intr. */
    }
    else
    {
		pDevice->tx_full = 1;
		flag = 0xe0000000;	/* Tx-done intr. */
    }
    if (entry == TX_RING_SIZE - 1)
		flag = 0xe0000000 | DESC_RING_WRAP;

    pDevice->tx_ring[entry].length = cpu_to_le32(pSTA_Module->MDL_TX_Len);
    pDevice->tx_ring[entry].length |= flag;

    if (entry == TX_RING_SIZE - 1)
    {
		pDevice->tx_ring[entry].length |= 0x02000000;  // end of ring
    }

    Status = pSTA_Module->MDL_Max_Basic_Data_Rate << 20;
    Status = Status + DescOwned + pDevice->STA_Module.MDL_Retry_Count;
    pDevice->tx_ring[entry].status = cpu_to_le32(Status);
    pDevice->cur_tx++;


    dev1->trans_start = jiffies;
    /* Trigger an immediate transmit demand. */
    outl(0, dev1->base_addr + CSR1);

}

/************************************************************************/
/* 			Read/Write CSR Function				*/
/************************************************************************/
void LM_Write_CSR(STA_MDL * pSTA_Module, BYTE CSR_Offset,unsigned long Value32)
{
    struct net_device *dev1 =
	(struct net_device *) pSTA_Module->MDL_pDrvObj;
    long ioaddr = dev1->base_addr;
    outl(Value32, ioaddr + CSR_Offset);
}

DWORD LM_Read_CSR(STA_MDL * pSTA_Module, BYTE CSR_Offset)
{
    DWORD R_Value;
    struct net_device *dev1 =
	(struct net_device *) pSTA_Module->MDL_pDrvObj;
    long ioaddr = dev1->base_addr;
    R_Value = inl(ioaddr + CSR_Offset);
    return R_Value;
}

/************************************************************************/
/*			Spin lock/unlock Function			*/
/*			should declare spinlock_t lock			*/
/*			spin_lock_init (&lock);				*/
/************************************************************************/
void LM_Spin_Lock()
{
    spin_lock(&lock);
}

void LM_Spin_Unlock()
{
    spin_unlock(&lock);
}

/************************************************************************/
/*				delay function: ms			*/
/************************************************************************/
void LM_Delay(WORD x)
{
	mdelay(x);
}

UINT8 LM_Random()
{
    BYTE Value8;
    get_random_bytes(&Value8, 1);
    return Value8;
}

void LM_Set_Channel(STA_MDL * pSTA_Module, BYTE channel)
{

        struct net_device *dev1 =(struct net_device *) pSTA_Module->MDL_pDrvObj;
	struct adm8201_private *pDevice = (struct adm8201_private *) dev1->priv;
	#ifdef	adm8201_debug
		printk(" set channel %d \n",channel);
	#endif

        RF_SetChannel(&(pDevice->LM_Device),channel);
}

void LM_SetAntenna(STA_MDL * pSTA_Module , int Antenna )
{
	struct net_device *dev1 =(struct net_device *) pSTA_Module->MDL_pDrvObj;
	struct adm8201_private *pDevice = (struct adm8201_private *) dev1->priv;

	#ifdef	lm_debug
		printk("Set Antenna = %x \n ",Antenna );
	#endif

	RF_SetAntenna(&(pDevice->LM_Device), Antenna );

}


void LM_Adm8201_Reset(struct net_device *dev)
{
    struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    long ioaddr = dev->base_addr;
    int i;

	#ifdef	lm_debug
		printk("Reset adapter \n");
	#endif
    netif_stop_queue(dev);

    empty_rings(dev);

    outl(0, ioaddr + CSR6);	/* for soft rest */

    outl(0, ioaddr + CSR7);	 // disable interrupt

    ChipReset(&(pDevice->LM_Device));

    adm8201_init_ring(dev);

    Hardware_Init(&(pDevice->LM_Device));

    /* Enable interrupts by setting the interrupt mask. */
    outl(0x040180c5 , ioaddr + CSR5);

    outl(pDevice->LM_Device.IntMaskBackUp , ioaddr+CSR7);

    outl(0x00002002, ioaddr + CSR6);

    outl(0, ioaddr + CSR2);	/* Rx poll demand */

    mdelay(100);

    if (pDevice->STA_Module.MDL_Operation_Mode == Ndis802_11IBSS)
    {
        for(i=0;i<pDevice->STA_Module.MDL_ADHOC_Mode.SSID[0];i++)
	    pDevice->STA_Module.MDL_SSID[i] =
		    pDevice->STA_Module.MDL_ADHOC_Mode.SSID[i];
    }
	UM_Init_Module(&pDevice->STA_Module);
	netif_start_queue(dev);

}
void LM_CheckDxferState(struct net_device *dev)
{
    	struct adm8201_private *pDevice = (struct adm8201_private *) dev->priv;
    	long ioaddr = dev->base_addr;
	BYTE	rxPkt1in;
	WORD	rra, rwa;
	DWORD	Value32;

	Value32 = inl(ioaddr + CSR_TEST1);
	rra = (WORD)((Value32>>12)&0x1ff);
	rwa = (WORD)((Value32>>2) &0x1ff);
	rxPkt1in = (BYTE)((Value32 & BIT_1)>>1);
	if( (rra != rwa) && (rxPkt1in == 0) )
	{
		netif_stop_queue(dev);
		LM_Adm8201_Reset(dev);
		pDevice->LM_Device.LostBeaconFlag = 0;
		netif_start_queue(dev);
	}//end if()
	
}//end LM_CheckDxferState()
